#ifndef dist_h
#define dist_h

#include <map>
#include <string>
#include <iostream>
#include <fstream>
#include <set>
#include <deque>
#include <list>
#include <sstream>
#include <vector>

#include <assert.h>

#include "dataobj.h"
#include "global.h"
//#include "debug.h"
#include "topk.h"

using namespace std;

class Dist {
    deque <string> objList;

    deque <string> refList;

    map <string, DataObj> id2obj;
    //map <int, string> int2id;
	map <string, string> intId2id;
    //map <string, int> id2int;
	map <string, string> id2intId;

    bool countDist;//if true count the number of total distance and use distBuffer to save distance computations
    map <string, float> distBuffer;//if countDist is true, use this buffer to save distance computations

	map < string, float> distBufferBak; //backup distBuffer;

	MObj * mo;//to load the m file. For the testing purpose
    
    /* compute the distance between two objects */
    float compDist(const string & idA, const string & idB);

public:
	/* Maximal float point error */
	static float fp_err;
    
	const static float maxDist;// = DIST_INFTY;
		//float maxDist;
	
		//static int str2hexFlag;
	
	Dist() {
		//maxDist = DIST_INFTY;
		init();
	}

	~Dist() {
		if (mo != nil) delete mo;
	}

	void init() {
		countDist = false;
		mo = nil;
	}

	map <string, float> & getDistBuf() {
		return distBuffer;
	}

	DataObj & getDataObj(string & id) {
		return id2obj[id];
	}

	/* translate int id to string id. The int id is the order specified in getAllDataObj */
	void toStrId(const deque <int> & inList, deque <string> & outList) {
		deque <string> ::iterator it;
		string * arr = new string[objList.size()];
		int i ;
		for(it = objList.begin(), i = 0; it != objList.end(); it ++, i ++) {
			string id = *it;
			arr[i] = id;
		}

		deque <int> ::const_iterator jt;
		for(jt = inList.begin(); jt != inList.end(); jt ++)
			outList.push_back(arr[*jt]);

		delete [] arr;
	}

	int getObjNum() {
		return objList.size();
	}

	/* get all DataObj that is in the objList */
	void getAllDataObj(deque <DataObj> & resList) {
		deque <string> ::iterator it;
		for(it = objList.begin(); it != objList.end(); it ++) {
			string id = *it;
			resList.push_back(id2obj[id]);
		}
	}

	/* convert a hex string id to a string id */
	string getStrId(string hexId) {
		return intId2id[hexId];
	}

	/* convert a id string to a hex string */
	string str2hexStr(string & str) {
		string intId = id2intId[str];
		return intId;
	}

	/* convert a deque of str id. 
		Return the result in strList 
	*/
	void str2hexStrBatch(deque <string> & strList) {
		deque <string> newList;
		deque <string> ::iterator it;
		for(it = strList.begin(); it != strList.end(); it ++) {
			string id = *it;
			newList.push_back(str2hexStr(id));
		}
		strList.clear();
		for(it = newList.begin(); it != newList.end(); it ++) {
			string newId = *it;
			strList.push_back(newId);
		}
	}

	/* backup distBuffer for future used */
	void backupDistBuffer() {
		distBufferBak = distBuffer;
	}

	/* restore distBuffer from backup */
	void restoreDistBuffer() {
		distBuffer = distBufferBak;
	}

	/* to remove a list of objects from objList */
	void exclude(deque <string> & objectsToExclude) {
		set <string> excludeSet;
		deque <string> ::iterator it;
		for(it = objectsToExclude.begin(); it != objectsToExclude.end(); it ++) {
			excludeSet.insert(*it);
		}
		deque <string> bakList;

		it = objList.begin(); 
		while(it != objList.end()){
			string ob = *it;
			set <string>::const_iterator cIt = excludeSet.find(ob);
			if(cIt != excludeSet.end()) {
				//deque <string> ::iterator toRemoveIt = it;
				it ++;
				//objList.erase(toRemoveIt);
			}else {
				it ++;
				bakList.push_back(ob);
			}
		}
		objList.clear();
		for(it = bakList.begin(); it != bakList.end(); it ++) {
			string ob = *it;
			objList.push_back(ob);
		}
	}

	/* for a given object q, get the number of neighbors within a radius r */
	int getNumNeighbors(const string & q, float r) {
		int count = 0;
		deque <string> ::const_iterator it;
        for(it = objList.begin(); it != objList.end(); it ++) {
            const string & ob = *it;
			float d = get(q,ob);
			if(d < r) 
				count ++;
        }
		return count;
	}

	void clear() {
		distBuffer.clear();
	}
    
	void print() {
		deque <string> ::const_iterator it;
		for(it = objList.begin(); it != objList.end(); it ++) {
			string id = *it;
			id2obj[id].print(cout);
		}
	}

    /* put the list of true top k-nn objects in resList and return the maximal distances among them*/
    float getTrueTopK(const string & queryOb, int k, set <string> & resList) {
        deque <string>::iterator it;
        TopKList trueList(k);
        for(it = objList.begin(); it != objList.end(); it ++) {
            string & ob = *it;
            if(queryOb != ob) {
                float d = get(queryOb, ob);
                trueList.insert(ob, d);
            }
        }
        trueList.getTopK(resList);
		float maxDist = trueList.getCurMaxPriority();

		for(it = objList.begin(); it != objList.end(); it ++) {
			string & ob = *it;
            if(queryOb != ob) {
                float d = get(queryOb, ob);
                if(d <= maxDist) 
					resList.insert(ob);
            }			
		}
		
		return maxDist;
    }

    /* read in the data 
		To do: figure out why dp can not be deleted.
	*/
    void readData(char * inFile) {
        ifstream is(inFile);
		if( ! is ) {
			cerr << "Error opening input stream "
				<< inFile << endl;
			//return;
			exit(1);
		}  

        std::string sLine;
        do{
            getline(is, sLine);
            if(sLine.length() > 0) {
                DataObj * dp = new DataObj(sLine);
				int sz = objList.size();
				char temp[sz];
				sprintf(temp, "%x", sz);
				string intId = temp;

				id2obj[intId] = * dp;
				intId2id[intId] = sLine;
                //id2int[sLine] = objList.size();
				id2intId[sLine] = intId;
                //objList.push_back(sLine);
				objList.push_back(intId);
            }
        }while(sLine.size() > 0);
        is.close();
    }
    
    /* insert the native structure to the list. Return the interal id.
     * strId: the readable text id of the native.
     */
    string insertNative(string & strId) {
    	string id = "";
        if(strId.length() > 0) {
            DataObj * dp = new DataObj(strId);

			int sz = objList.size();
			char temp[sz];
			sprintf(temp, "%x", sz);
			string intId = temp;

			id2obj[intId] = * dp;
			intId2id[intId] = strId;
			id2intId[strId] = intId;
			//objList.push_back(intId);
			id = intId;
        }
        return id;
    }

	bool getCountDist() const {
		return countDist;
	}

    /* set countDist to count the total number of accessed distance */
    void setCountDist(bool val) {
        countDist = val;
    }
    
    /* get the number of accessed distances */
    int getNumAccessed() const {
        assert(countDist == true);
        return distBuffer.size();
    }
    
    /* get the list of all data object IDs */
    deque <string> & getAllObjects() {
        return objList;
    }

	/** split buffer id. 
	s: a dist buffer id.
	*/
	void splitBufId(vector <string> & result, string & s) {
		Pattern::split(result, s, "-");
	}

	/* Precompute the distances between a list of reference objects and all data objects.
	 * Store the results in the distance buffer.
	 */
	void preCompute(deque <string> & refList) {
		deque <string> ::iterator it;
		for (it = refList.begin(); it != refList.end(); it ++) {
			string idB = *it;
			deque <string> ::iterator jt;
			for (jt = objList.begin(); jt != objList.end(); jt ++) {
				string idA = *jt;
				string bufferId;
				////if(intIdA <= intIdB)
				////	bufferId = intIdA + "-" + intIdB;
				////else bufferId = intIdB + "-" + intIdA;

#ifdef SYMMETRIC
		if(idA <= idB)
			bufferId = idA + "-" + idB;
		else bufferId = idB + "-" + idA;
#else
		bufferId = idA + "-" + idB;
#endif

				float d = compDist(idA, idB);
				distBuffer[bufferId] = d;
			}
		}
	}

	/* number of neighbor with a given radius */
	int neighborNum(string id, float radius) {
		int count = 0;
		deque <string> ::iterator it;
		for (it = objList.begin(); it != objList.end(); it ++) {
			string ob = *it;
			float d = get(id, ob);
			if (d < radius) count ++;
		}
		return count;
	}

	/* Get the total distances to all objects
	 *
	 */
	double totalDist(string & id) {
		double sum = 0;
		deque <string> ::iterator it;
		for (it = objList.begin(); it != objList.end(); it ++) {
			string ob = *it;
			float d = get(id, ob);
			sum += d;
		}
		return sum / objList.size();
	}

	/* Return the median distance to all object */
	float medianTotalDist(string & id) {
		list <strFloatPair> temp;
		deque <string> ::iterator it;
		for (it = objList.begin(); it != objList.end(); it ++) {
			string ob = *it;
			float d = get(id, ob);
			strFloatPair sfp(ob, d);
			temp.push_back(sfp);
		}
		temp.sort(SortProcess());

		// Get median
		int median = objList.size() / 2;
		int i = 0;
		list <strFloatPair> ::iterator jt = temp.begin();
		while (jt != temp.end() && i < median) {
			jt ++;
			i ++;
		}
		return jt->second;
	}

	/* collect the top 10 best decoys */
	void setBestDecoys(Global & gb) {
		int i;
		//if (gb.methodId == USER_LOCAL_CENTER) {
		switch (gb.methodId) {
		case USER_LOCAL_CENTER:
			{
				deque <string> tempList;
				gb.readList(gb.bestFile, tempList);
				deque <string> ::iterator tt;
				for (tt = tempList.begin(), i = 0; tt != tempList.end() && i < 10; tt ++, i++) {
					string strId = *tt;
					string id = str2hexStr(strId);
					gb.bestList.push_back(id);
				}
			}
			break;
		default:
			{
				list <strFloatPair> sfpList;
				deque <string> ::iterator it;
				for (it = objList.begin(); it != objList.end(); it ++) {
					string ob = *it;
					float val;
					if (gb.methodId == REF_LOCAL_CENTER) {
						val = gb.ob2rd[ob];
					}else {
						string strId = getStrId(ob);
						string energy = gb.getEnergy(strId);
						if (energy != "N/A")
							val = atof(energy.c_str());
						else val = 999999.9;
					}
					strFloatPair sfp(ob, val);
					sfpList.push_back(sfp);
				}
				sfpList.sort(SortProcess());

				list <strFloatPair> ::iterator jt;

				for (jt =sfpList.begin(), i = 0; jt != sfpList.end() && i < 10; jt ++, i++) {
					gb.bestList.push_back(jt->first);
				}
			}
		}
	}

	/* distances to top 10 best structures */
	float bestTotalDist(Global & gb, string & id) {
		vector <string> & bestList = gb.bestList;
		if (bestList.empty()) {
			setBestDecoys(gb);
		}

		double ls = 0;
		vector <string> ::iterator it;
		for (it = bestList.begin(); it != bestList.end(); it ++) {
			string ob = *it;
			float d = get(ob, id);
			ls += d;
		}
		return ls / bestList.size();
	}

    /* get the distance between two objects */
    float get(const string & idA, const string & idB) {
		////string & intIdA = id2intId[idA];
		////string & intIdB = id2intId[idB];
    	float retVal = maxDist;
        if(countDist) {
            string bufferId;
			////if(intIdA <= intIdB)
			////	bufferId = intIdA + "-" + intIdB;
			////else bufferId = intIdB + "-" + intIdA;

#ifdef SYMMETRIC
			if(idA <= idB)
				bufferId = idA + "-" + idB;
			else bufferId = idB + "-" + idA;
#else
            bufferId = idA + "-" + idB;
#endif
            map <string, float> :: const_iterator it = distBuffer.find(bufferId);
            if(it == distBuffer.end()) {
                //float d = compDist(idA, idB);
				//float d = compDist(intIdA, intIdB);
				float d = compDist(idA, idB);
                distBuffer[bufferId] = d;
                retVal = d;
			}else {
				retVal = distBuffer[bufferId];
			}
        }else retVal = compDist(idA, idB);

#ifdef SYMMETRIC
#else
        if (retVal > maxDist - 1) {
        	cout << "Error! Accessing a distance in the wrong order!";
        	exit(1);
        }
#endif
        return retVal;
    }

	/* get k-nn distance */
	float getKnnDist(int k, string & queryOb, set <string> & candidates) {
		TopKList topK(k);
		set <string> ::const_iterator it;
		for(it = candidates.begin(); it != candidates.end(); it ++) {
			string ob = *it;
			float d = get(queryOb, ob);
			topK.insert(ob, d);
		}
		return topK.getCurMaxPriority();
	}

	/* get k-nn distance */
	float getKnnDist(int k, string & queryOb, deque <string> & candidates) {
		TopKList topK(k);
		deque <string> ::const_iterator it;
		for(it = candidates.begin(); it != candidates.end(); it ++) {
			string ob = *it;
			float d = get(queryOb, ob);
			topK.insert(ob, d);
		}
		return topK.getCurMaxPriority();
	}

	/* clear mo */
	void clearMo() {
		if(mo != null) {
			delete mo;
			mo = null;
		}
	}

	/* (For testing purpose) Read in an m file and store the distances in mo. */
	void useMFile(char inFile[]) {
		if(mo != null) {
			printf("Error! The M file has already been loaded!");
			exit(1);
		}
		mo = new MObj();
		mo->readMFile(inFile);
	}

    /* get a list of random objects that can not be in the excludeList and return then in resList */
    void getRandList(unsigned objNum, const deque <string> & excludeList, deque <string> & resList) {
        //build an exclusion set
        set <string> excSet;
        deque <string> :: const_iterator eIt;
        for(eIt = excludeList.begin(); eIt != excludeList.end(); eIt ++) {
            string id = *eIt;
			excSet.insert(id);
        }

		list <string> tempList;
		deque <string> ::const_iterator it;
		for(it = objList.begin(); it != objList.end(); it ++) {
			string id = *it;
			set <string> ::const_iterator rIt = excSet.find(id);
			if(rIt == excSet.end()) {
				tempList.push_back(id);
			}
		}
        
        //select the random set
		while(resList.size() < objNum) {
			int index = rnd(0, tempList.size()-1);
			int i = 0;
			list <string> ::iterator lIt = tempList.begin();
			while(i != index) {
				assert(lIt != tempList.end());
				i ++;
				lIt ++;
			}
			assert(lIt != tempList.end());
			string id = *lIt;
			resList.push_back(id);
			tempList.erase(lIt);
		}
    }
    
    /* get a random object */
    string getRandomObj() {
        int index = rnd(0, objList.size()-1);
		char temp[10];//ToDo: 10 can be reduce to optimize performance
		sprintf(temp, "%x", index);
		string res = temp;
		return res;
    }

};

#endif







