#ifndef distdb_h
#define distdb_h

#include <assert.h>
#include <deque>
#include <math.h>
#include <set>
#include <sstream>

#include "dist.h"
//#include "debug.h"
#include "topk.h"
#include "global.h"
#include "forest.h"

using namespace std;


class DistDb {
    Dist dist; // Contains all distance information.
    deque <string> refList; // list of pivots/references.
public:

	/* convert a deque of str id. 
		Return the result in strList 
	*/
	void str2hexStrBatch(deque <string> & strList) {
		dist.str2hexStrBatch(strList);
	}

    /* set countDist in dist to count the total number of accessed distance */
    void setCountDist(bool val) {
		dist.setCountDist(val);
    }

	/* backup distBuffer */
	void backupDistBuffer() {
		dist.backupDistBuffer();
	}

	/* remove a list of object from dist */
	void exclude(deque <string> & objectsToExclude) {
		dist.exclude(objectsToExclude);
	}

	/* Divide all objects into multiple bins.
	 * Return the mapping of objects to hex string in refHash.
	 * p: how many bins to use in the hashing.
	 * refList: the list of reference objects to use.
	 * objList: the list of objects to hash
	 * refHash: map a object to a hex number, using a set of pivots.
	 */
	void multiHash(int p, deque <string> & refList, deque <string> & objList,
			map <string, string> & refHash) {
		string binLabels[p];
		string temp = "0123456789abcdef";
		for (int i = 0; i < p; i ++) {
			binLabels[i] = temp.substr(i,1);
		}

		deque <string> ::iterator ot;
		for(ot = objList.begin(); ot != objList.end(); ot ++) {
			string ob = *ot;
			refHash[ob] = "";
			gb.ob2rd[ob] = 0;
		}

		deque <string> ::iterator it;
		for (it = refList.begin(); it != refList.end(); it ++) {
			string ref = *it;
			list <strFloatPair> tempList;

			deque <string> ::iterator jt;
			for (jt = objList.begin(); jt != objList.end(); jt ++) {
				string ob = *jt;
				double d = dist.get(ob, ref);
				gb.ob2rd[ob] += d;
				strFloatPair sfp(ob, d);
				tempList.push_back(sfp);
			}

			tempList.sort(SortProcess());

			// Assign the bin labels.
			int objNum = objList.size();
			int step = (int)ceil(objNum / (float)p);
			int count = 0;
			list <strFloatPair> ::iterator st;
			int index = 0;
			for (st = tempList.begin(); st != tempList.end(); st ++) {
				if (count == step) {
					index ++;
					count = 0;
				}else count ++;
				assert(index >=0 && index < 16);
				refHash[st->first] += binLabels[index];
			}
		}
	}

	/* Hash objects to binary strings according to the median object.
	 * Return the mapping of objects to binary string in refHash.
	 * refList: the list of reference objects to use.
	 * objList: the list of objects to hash
	 * refHash: For each pivot, map a object to 0 or 1.
	 */
	void medHash(deque <string> & refList, deque <string> & objList,
			map <string, string> & refHash) {
		int objNum = objList.size();
		int medium = int(objNum / 2);

		deque <string> ::iterator ot;
		for(ot = objList.begin(); ot != objList.end(); ot ++) {
			string ob = *ot;
			refHash[ob] = "";
		}

		deque <string> ::iterator it;
		for (it = refList.begin(); it != refList.end(); it ++) {
			string ref = *it;
			list <strFloatPair> tempList;

			deque <string> ::iterator jt;
			for (jt = objList.begin(); jt != objList.end(); jt ++) {
				string ob = *jt;
				double d = dist.get(ob, ref);
				strFloatPair sfp(ob, d);
				tempList.push_back(sfp);
			}

			tempList.sort(SortProcess());

			int count;
			list <strFloatPair> ::iterator st;
			for (st = tempList.begin(), count = 0; st != tempList.end(); st ++, count ++) {
				if (count < medium) refHash[st->first] += "0";
				else refHash[st->first] += "1";
			}
		}
	}

	/* split a binary string to two hash tables
	 * ob: The object to map.
	 * bStr: The binary string.
	 * intList: the first half othe queue determines which bits will be in the
	 * first hash table, and second half will be in the second hash table.
	 */
	void splitStr(string & ob, string & bStr, deque <int> & intList,
			map <string, string> & refHash1,
			map <string, string> & refHash2) {
		int size = bStr.size();
		int mid = int(size / 2);
		string s1 = "";
		string s2 = "";
		deque <int> ::iterator it;
		int i;
		for (it = intList.begin(), i = 0; it != intList.end(); it ++, i ++) {
			int index = *it;
			char bit = bStr[index];
			if (i < mid) {
				s1 += bit;
			}else {
				s2 += bit;
			}
		}
		refHash1[ob] = s1;
		refHash2[ob] = s2;
	}

	/* Create a list of integers from begin to end (inclusive), with the order randomly shuffled.
	 * intList: the returned result.
	 */
	void rndIntList(int begin, int end, deque <int> & intList) {
		unsigned size = end - begin + 1;

		list <int> tempList;
		for (int i = begin; i <= end; i++) {
			tempList.push_back(i);
		}

		while (intList.size() < size) {
			int r = rnd(0, tempList.size() - 1);
			list <int> ::iterator it = tempList.begin();
			for (int j = 0; j < r; j ++) it ++;
			intList.push_back(*it);
			tempList.erase(it);
		}
	}

	/* Output pairwise distances to references for top objects */
	void outPwDist2Ref(Global & gb) {
		deque <string> & objList = dist.getAllObjects();

		list <strFloatPair> tempList;
		deque <string> ::iterator it;
		for (it = objList.begin(); it != objList.end(); it ++) {
			string ob = *it;
			double ls = 0;
			deque <string> ::iterator rt;
			for (rt = refList.begin(); rt != refList.end(); rt ++) {
				string r = *rt;
				float d = dist.get(ob, r);
				ls += d;
			}

			strFloatPair sfp(ob, ls);
			tempList.push_back(sfp);
		}
		tempList.sort(SortProcess());

		list <strFloatPair> ::iterator tt;
		int i;
		//for (tt = tempList.begin(), i = 0; tt != tempList.end() && i < gb.sampleSize; tt ++, i ++) {
		for (tt = tempList.begin(), i = 0; tt != tempList.end(); tt ++, i ++) {
			string id = tt->first;
			string strId = dist.getStrId(id);
			string rmsd = gb.getClass(strId);
			string energy = gb.getEnergy(strId);
			cout << "outR " << strId;
			cout << " ls " << tt->second;
			cout << " rmsd " << rmsd;
			cout << " energy " << energy;
			cout << endl;
		}
	}

	/* Compute pairwise distances to references */
	void pwDist2Ref(Global & gb) {
		deque <string> & objList = dist.getAllObjects();

		deque <string> ::iterator it;
		for (it = objList.begin(); it != objList.end(); it ++) {
			string ob = *it;
			double ls = 0;
			deque <string> ::iterator rt;
			for (rt = refList.begin(); rt != refList.end(); rt ++) {
				string r = *rt;
				float d = dist.get(ob, r);
				ls += d;
			}

			gb.ob2rd[ob] = ls;
		}
	}

	/* Estimate density using the median object to construct the hashing table,
	 * and perform hashing to remove dissimilar samples. When estimating the
	 * density, use HS-Forest. In the 2D map of height and size of neighborhood,
	 * pick the diagonal one.
	 * Return a list of object that has high density around them in resList.
	 */
	void forestDiag(Global & gb, deque <string> & resList) {
		deque <string> & objList = dist.getAllObjects();

		int numTrees = gb.treeNum;
		int hashSize = gb.hashSize;

		map <string, string> & refHash = gb.getRefHash();
		multiHash(gb.splitFactor, refList, objList, refHash);

		Forest f;
		set <string> samples;
		for (int i = 0; i < numTrees; i ++) {
			Tree * t = new Tree(gb.denThres);
			t->setLeafThres(gb.leafThres);
			t->initRefArr(hashSize);
			f.addTree(t);

			int * refArr = t->getRefArr();
			getRandInt(refList.size(), hashSize, refArr);
			t->buildTree(gb, refHash, objList, refArr, hashSize);
		}

		switch (gb.methodId) {
		case USER_LOCAL_CENTER:
		case REF_LOCAL_CENTER:
		case BD_LOCAL_CENTER:
			f.localCenter(dist, gb, resList);
			break;
		default:
			cout << "Wrong method ID!" << endl;
			exit(1);
		}
	}

	/* Return a list of objects sampled uniformly.
	 */
	void ranSam(int sampleSize, deque <string> & resList) {
		deque <string> & objList = dist.getAllObjects();
		getRandListFromList(sampleSize, objList, resList);
	}
    
	/* output the dense objects to a file */
	void outDenseObjects(deque <string> & outList, char outFileName[]) {
		ofstream os(outFileName);
		outQ(outList, os);
		os.close();
	}

    /* translate and output a queue to IO. Write both the internal id, the class label,
     * and the label statistics.
     * translated ID and the class information.
     */
	void outQ2col(deque <string> & outList, Global & gb, ostream & os) {
		map <string, int> labelStat;
		deque <string> ::iterator it;
		for (it = outList.begin(); it != outList.end(); it ++) {
			string ob = *it;
			string newId = dist.getStrId(ob);
			//os << newId << " internal: " << ob;
			string label = gb.getClass(newId);
			//os << " class: " << label <<endl;

			// Collect label statistics.
			map <string, int> ::iterator mIt = labelStat.find(label);
			if (mIt == labelStat.end()) {
				labelStat[label] = 1;
			}else labelStat[label] ++;
		}

		// Output label statistics.
		//os << endl << endl << "Statistics:" << endl;
		map <string, int> ::iterator mIt;
		for (mIt = labelStat.begin(); mIt != labelStat.end(); mIt ++) {
			os << "classStat: " << mIt->first << ": " << mIt->second << endl;
		}
	}

    /* translate and output a queue to IO. Write both the internal id and the
     * translated ID.
     */
    void outQ2col(deque <string> & outList, ostream & os) {
    	deque <string> ::iterator it;
    	for (it = outList.begin(); it != outList.end(); it ++) {
    		string ob = *it;
    		string newId = dist.getStrId(ob);
    		os << newId << " internal: " << ob << endl;
    	}
    }

    /* translate and output a queue to IO */
    void outQ(deque <string> & outList, ostream & os) {
    	deque <string> ::iterator it;
    	for (it = outList.begin(); it != outList.end(); it ++) {
    		string ob = *it;
    		string newId = dist.getStrId(ob);
    		os << newId << endl;
    	}
    }

	/** output a set to screen */
	void outSet(set <string> & canList) {
		set <string>::iterator it;
		int i = 1;
		for(it = canList.begin(); it != canList.end(); it ++) {
			cout << i << ": " << *it << endl;
			i ++;
		}
	}

	/** output a set to screen */
	void outSet(string & queryOb, Dist & dist, set <string> & canList) {
		set <string>::iterator it;
		int i = 1;
		for(it = canList.begin(); it != canList.end(); it ++) {
			string c = *it;
			cout << i << ": " << c << " dist " << dist.get(queryOb, c) << endl;
			i ++;
		}
	}  

	/** output a list to screen */
	void outSet(list <string> & canList) {
		list <string>::iterator it;
		int i = 1;
		for(it = canList.begin(); it != canList.end(); it ++) {
		cout << i << ": " << *it << endl;
		i ++;
		}
	}

	/** output a list to screen */
	void outSet(string & queryOb, Dist & dist, list <string> & canList) {
		list <string>::iterator it;
		int i = 1;
		for(it = canList.begin(); it != canList.end(); it ++) {
			string c = *it;
			cout << i << ": " << c << " dist " << dist.get(queryOb, c) << endl;
			i ++;
		}
	} 

	/** output a set to screen */
	void outSet(string & queryOb, Dist & dist, set <string> & canList, ostream & os) {
		set <string>::iterator it;
		int i = 1;
		std::vector<std::string> temp;
		string queryObId = dist.getStrId(queryOb);
		//boost::split(temp, queryObId, boost::is_any_of("/"), boost::token_compress_on);
		Pattern::split(temp, queryObId, "/");
		os << "## Query object: " << temp[temp.size() - 1] << endl;
		for(it = canList.begin(); it != canList.end(); it ++) {
			string c = *it;
			string strId = dist.getStrId(c);
			//boost::split(temp, strId, boost::is_any_of("/"), boost::token_compress_on);
			Pattern::split(temp, strId, "/");
			os <<"## "<< i << ": " << temp[temp.size() - 1] << " dist " << dist.get(queryOb, c) << endl;
			i ++;
		}
	} 

    /* get the reference object list for reading or editing */
    deque <string> & getRefList() {
        return refList;
    }

	Dist & getDist() {
		return dist;
	}

	DataObj & getDataObj(string & id) {
		return dist.getDataObj(id);
	}

	/* translate int id to string id. The int id is the order specified in getAllDataObj */
	void toStrId(const deque <int> & inList, deque <string> & outList) {
		dist.toStrId(inList, outList);
	}

	void getAllDataObj(deque <DataObj> & resList) {
		dist.getAllDataObj(resList);
	}

	deque <string> & getAllObjects() {
		return dist.getAllObjects();
	}

	/** put all objects except those in exclusion in objSet */
	void getObjects(deque <string> & exclusion, set <string> & objSet) {
		deque <string> & obList = dist.getAllObjects();
		deque <string> ::iterator it;
		for(it = obList.begin(); it != obList.end(); it ++) {
			string s = *it;
			objSet.insert(s);
		}
		//deque <string>::iterator it;
		for(it = exclusion.begin(); it != exclusion.end(); it ++) {
			string s = *it;
			set <string>::iterator jt = objSet.find(s);
			if(jt != objSet.end()) objSet.erase(jt);
		}
	}

	/* Get a list of non-redundant random int in the range of [0, maxNum)
	 * samNum: number of random int to generates
	 * refArr: return the result list here.
	 */
	void getRandInt(int maxNum, int samNum, int * refArr) {
		// Build a temporal list
		bool isRemain[maxNum];// Indicate whether the index of the cell is in the remain list
				// or not.
		for (int i = 0; i < maxNum; i ++) isRemain[i] = true;

		// Select the random set
		int curSize = 0;
		while ( curSize < samNum) {
			int remainNum = maxNum - curSize;
			int index = rnd(0,remainNum - 1);// Random index

			int count = 0;
			int i = 0;
			while (i < maxNum) {
				if (isRemain[i]) {
					if (count == index) {
						refArr[curSize] = i;
						isRemain[i] = false;
						curSize ++;
						break;
					}
					count ++;
				}
				i ++;
			}
		}
	}

    /* get a list of random object */
    void getRandList(int objNum, deque <string> & resList) {
        deque <string> excList;//the exclusion list is empty
        dist.getRandList(objNum, excList, resList);
    }

    /* get a list of random reference objects in inList
	and return then in resList.
	resList: the result set. Must be empty at the begining
	remaining: the remaining objects in inList is put
	in remaining. Must be empty at the begining
	*/
    static void getRandListAndRemain(int objNum, deque <string> & inList,
		deque <string> & resList, deque <string> & remaining);

    /* get a list of random reference objects in inList and return then in resList
     * objNum: number of samples
     */
    static void getRandListFromList(int objNum, const deque <string> & inList,
		deque <string> & resList);

    /* get a list of random reference objects in inList and return them in resList */
    void getRandList(int objNum, deque <string> & inList, deque <string> & resList);

    /* get a list of random reference objects in inList that can not be in the excludeList and return then in resList */
    void getRandList(int objNum, deque <string> & excludeList,
		deque <string> & inList, deque <string> & resList);

    /* get a random object */
    string getRandomObj() {
        return dist.getRandomObj();
    }

	void clearRefList() {
		refList.clear();
	}

    void readData(char * inFile) {
        //cout << "Enter DistDb. To create a DistDb object.." << endl;
        dist.readData(inFile);
        //cout << "Exit DistDb" << endl;
    }

	/* set refList using a list of object */
	void setRefList(const deque <string> & inList) {
		deque <string> ::const_iterator it;
		for(it = inList.begin(); it != inList.end(); it ++)
			refList.push_back(*it);
	}
    
    /* get a refList by picking up the reference objects randomly*/
    void setRandRefList(int refNum, deque <string> & queryList) {
        dist.getRandList(refNum, queryList, refList);
    }
};

#endif


